package easik.sketch.attribute;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import javax.swing.tree.DefaultMutableTreeNode;

import easik.sketch.vertex.EntityNode;


/**
 * Class used to keep track of attributes used in unique key constraints.
 * 
 * @author Kevin Green 2006
 * @since 2006-06-06 Kevin Green
 * @version 2006-07-13 Kevin Green
 */
public class UniqueKey {

	/**
	 * The attributes contributing to this unique key
	 */
	private ArrayList<EntityAttribute> _attributes;
	/**
	 * The name of this key
	 */
	private String _keyName;
	/**
	 * The entity node to which this key applies
	 */
	private EntityNode _parent;
	/**
	 * The DefaultMutableTreeNode which represents this key graphically on the information tree
	 */
	private DefaultMutableTreeNode _uniqueKeyNode;
	
	/**
	 * Default constructor
	 * @param inParent The EntityNode to which this unique key applies
	 * @param inAtts An ArrayList of the attributes used in the key
	 * @param inName The name of the key
	 */
	public UniqueKey(EntityNode inParent, ArrayList<EntityAttribute> inAtts, String inName){
		_parent = inParent;
		_attributes = inAtts;
		_keyName = inName;
	}
	
	/**
	 * Overwriten method returns attribute list separated by semicolons
	 * 
	 * @return The name of this unique key formatted for use on the information tree
	 */
	public String toString(){
		String id = getId();
		id = _keyName + " (" + id + ")";
		return id;
	}
	
	/**
	 * Gets the id of this unique key.  The id is formatted by getting the name of each attribute and placing
	 * a semicolon between each name
	 * @return The id of this attribute
	 */
	public String getId(){
		String id = "";
		for(int i=0; i<_attributes.size(); i++){
			id += _attributes.get(i).getName() + ";";
		}
		id = id.substring(0,(id.length()-1));
		return id;
	}
	
	/**
	 * Returns an ArrayList of the attributes used in the key
	 * 
	 * @return An ArrayList of the attributes used in the key
	 */
	public ArrayList getAttributes(){
		return _attributes;
	}
	
	/**
	 * Sets the attributes involved in this unique key to the attributes present in <it>inAtts</it>
	 * @param inAtts Arraylist of EntityAttribute
	 */
	public void setAttributes(ArrayList<EntityAttribute> inAtts){
		_attributes = inAtts;
	}
	
	/**
	 * Returns the name of the key
	 * 
	 * @return The name of the key
	 */
	public String getKeyName(){
		return _keyName;
	}
	
	/**
	 * Sets the key name to the name described in <it>inName</it>
	 * @param inName The name of this attribute
	 */
	public void setKeyName(String inName){
		_keyName = inName;
	}
	
	/**
	 * Returns the parent entity for the key
	 * 
	 * @return The parent entity for the key
	 */
	public EntityNode getParent(){
		return _parent;
	}
	
	/**
	 * Sets the node used to display the unique key in the tree
	 * 
	 * @param inNode The node used to display the unique key in the tree
	 */
	public void setNode(DefaultMutableTreeNode inNode){
		_uniqueKeyNode = inNode;
	}
	
	/**
	 * Returns the node used to display the unique key in the tree
	 * 
	 * @return The node used to display the unique key in the tree
	 */
	public DefaultMutableTreeNode getNode(){
		return _uniqueKeyNode;
	}
	
	/**
	 * Returns if the name of a key is already used
	 * 
	 * @param inNode The entity being checked against
	 * @param keyName The name of the key
	 * @return If the key name is used or not
	 */
	public static boolean isKeyNameUsed(EntityNode inNode, String keyName){
		for(int i=0; i<inNode.getUniqueKeys().size(); i++){
			if(((UniqueKey)inNode.getUniqueKeys().get(i)).getKeyName().toUpperCase().equals(keyName.toUpperCase()))
				return true;
		}
		return false;
	}
	
	/**
	 * Returns if the key is unique or if another key exists with the exact same attributes
	 * 
	 * @param inNode The entity being checked against
	 * @param keyID The list of attributes in the key
	 * @return If the key is unique or if another key exists with the exact same attributes
	 */
	public static boolean isKeyUnique(EntityNode inNode, String keyID){
		for(int i=0; i<inNode.getUniqueKeys().size(); i++){
			if(((UniqueKey)inNode.getUniqueKeys().get(i)).getId().equals(keyID))
				return false;
		}
		return true;
	}
	
	/**
	 * Tests to see if there is any duplicate unique keys
	 * 
	 * @param inNode The node being tested
	 */
	public static void testUnique(EntityNode inNode){
		//Check for empty unique keys
		ArrayList emptyKeys = new ArrayList();
		for(int i=0; i<inNode.getUniqueKeys().size(); i++){
			UniqueKey curKey = (UniqueKey) inNode.getUniqueKeys().get(i);
			if(curKey.getAttributes().size() == 0){
				emptyKeys.add(curKey);
			}
		}
		int numEmpty = emptyKeys.size();
		while(emptyKeys.size() > 0){
			UniqueKey remKey = (UniqueKey) emptyKeys.remove(0);
			inNode.getUniqueKeys().remove(remKey);
			remKey.getNode().removeFromParent();
		}
		
		//Check for duplicates
		HashMap isDuplicate = new HashMap();
		for(int i=0; i<inNode.getUniqueKeys().size(); i++){
			UniqueKey curKey = (UniqueKey) inNode.getUniqueKeys().get(i);
			for(int j=i+1; j<inNode.getUniqueKeys().size(); j++){
				UniqueKey myKey = (UniqueKey) inNode.getUniqueKeys().get(j);
				if(curKey.getAttributes().equals(myKey.getAttributes()) && !isDuplicate.containsValue(myKey)){
					isDuplicate.put(myKey.getKeyName(), myKey);
				}
			}
		}
		int numDuplicate = isDuplicate.size();
		Iterator it = isDuplicate.keySet().iterator();
		while(it.hasNext()){
			UniqueKey remKey = (UniqueKey) isDuplicate.get(it.next());
			inNode.getUniqueKeys().remove(remKey);
			remKey.getNode().removeFromParent();
		}
	}
	
}
